import torch 
import torch.nn as nn 
import torch.nn.functional as F
from .binarization import *


    
class LeNet_5_cifar10_Masked(nn.Module):
    def __init__(self):
        super().__init__()

        self.conv1 = MaskedConv2d(3, 6, (5, 5), 1)
        self.conv2 = MaskedConv2d(6, 16, (5, 5), 1)
        self.fc3 = MaskedMLP(16 * 5 * 5, 60)
        self.fc4 = MaskedMLP(60, 30)
        self.fc5 = MaskedMLP(30, 10)

    def forward(self, x):
        x = F.relu(self.conv1(x))
        x = F.max_pool2d(x, 2, 2)
        x = F.relu(self.conv2(x))
        x = F.max_pool2d(x, 2, 2)

        x = F.relu(self.fc3(x.view(-1, 16 * 5 * 5)  )  )
        x = F.relu(self.fc4(x))
        
        return self.fc5(x)




class LeNet_5_cifar10(nn.Module):
    def __init__(self):
        super().__init__()

        self.conv1 = nn.Conv2d(3, 6, (5, 5), 1, bias = False)
        self.conv2 = nn.Conv2d(6, 16, (5, 5), 1, bias = False)
        self.fc3 = nn.Linear(16 * 5 * 5, 60, bias = False)
        self.fc4 = nn.Linear(60, 30, bias = False)
        self.fc5 = nn.Linear(30, 10, bias = False)

    def forward(self, x):
        x = F.relu(self.conv1(x))
        x = F.max_pool2d(x, 2, 2)
        x = F.relu(self.conv2(x))
        x = F.max_pool2d(x, 2, 2)

        x = F.relu(self.fc3(x.view(-1, 16 * 5 * 5)  )  )
        x = F.relu(self.fc4(x))
        
        return self.fc5(x)
    
class LeNet_5_cifar100_masked(nn.Module):
    def __init__(self):
        super(LeNet_5_cifar100_masked, self).__init__()
        self.conv1 = MaskedConv2d(3, 6, (5,5),1)
        self.conv2 = MaskedConv2d(6, 16, (5,5), 1)
        self.fc3 = MaskedMLP(16 * 5 * 5, 120)
        self.fc4 = MaskedMLP(120, 84)
        self.fc5 = MaskedMLP(84, 100)

    def forward(self, x):
        x = F.relu(self.conv1(x))
        x = F.max_pool2d(x, 2, 2)
        x = F.relu(self.conv2(x))
        x = F.max_pool2d(x, 2, 2)

        x = F.relu(self.fc3(x.view(-1, 16 * 5 * 5)  )  )
        x = F.relu(self.fc4(x))
        
        return self.fc5(x)
    
class LeNet_5_cifar100(nn.Module):
    def __init__(self):
        super().__init__()
        
        super(LeNet_5_cifar100, self).__init__()
        self.conv1 = nn.Conv2d(3, 6, (5,5),1 , bias = False)
        self.conv2 = nn.Conv2d(6, 16, (5,5), 1 , bias = False)
        self.fc3 = nn.Linear(16 * 5 * 5, 120, bias = False)
        self.fc4 = nn.Linear(120, 84, bias = False)
        self.fc5 = nn.Linear(84, 100, bias = False)

    def forward(self, x):
        x = F.relu(self.conv1(x))
        x = F.max_pool2d(x, 2, 2)
        x = F.relu(self.conv2(x))
        x = F.max_pool2d(x, 2, 2)

        x = F.relu(self.fc3(x.view(-1, 16 * 5 * 5)  )  )
        x = F.relu(self.fc4(x))
        
        return self.fc5(x)
    
class masked_Conv4_BN(nn.Module):

    def __init__(self):
        super().__init__()

        self.conv1 = MaskedConv2d(3, 64,(5,5), stride=1, padding=0)
        self.bn1 = nn.BatchNorm2d(64)
        self.conv2 = MaskedConv2d(64, 64, (5,5), stride=1, padding=0)
        self.bn2 = nn.BatchNorm2d(64)
        self.conv3 = MaskedConv2d(64, 128, (5,5), stride=1, padding=0)
        self.bn3 = nn.BatchNorm2d(128)
        self.conv4 = MaskedConv2d(128, 128, (5,5), stride=1, padding=0)
        self.bn4 = nn.BatchNorm2d(128)

        self.dense1 = MaskedMLP(512, 128)
        self.bn5 = nn.BatchNorm1d(128)
        self.dense2 = MaskedMLP(128, 128)
        self.bn6 = nn.BatchNorm1d(128)
        self.dense3 = MaskedMLP(128, 10)

    def forward(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = F.max_pool2d(F.relu(self.bn2(self.conv2(x))), kernel_size=2, stride=2)
        x = F.relu(self.bn3(self.conv3(x)))
        x = F.max_pool2d(F.relu(self.bn4(self.conv4(x))), kernel_size=2, stride=2)
        x = x.view(x.size(0), -1)
        x = F.relu(self.bn5(self.dense1(x)))
        x = F.relu(self.bn6(self.dense2(x)))
        x = self.dense3(x)
        return x
    
class Conv4_BN(nn.Module):

    def __init__(self):
        super().__init__()
        self.conv1 = nn.Conv2d(3, 64, kernel_size=5, stride=1, padding=0, bias = False)
        self.bn1 = nn.BatchNorm2d(64)
        self.conv2 = nn.Conv2d(64, 64, kernel_size=5, stride=1, padding=0, bias = False)
        self.bn2 = nn.BatchNorm2d(64)
        self.conv3 = nn.Conv2d(64, 128, kernel_size=5, stride=1, padding=0, bias = False)
        self.bn3 = nn.BatchNorm2d(128)
        self.conv4 = nn.Conv2d(128, 128, kernel_size=5, stride=1, padding=0, bias = False)
        self.bn4 = nn.BatchNorm2d(128)

        self.dense1 = nn.Linear(512, 128, bias = False)
        self.bn5 = nn.BatchNorm1d(128)
        self.dense2 = nn.Linear(128, 128, bias = False)
        self.bn6 = nn.BatchNorm1d(128)
        self.dense3 = nn.Linear(128, 10, bias = False)

    def forward(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = F.max_pool2d(F.relu(self.bn2(self.conv2(x))), kernel_size=2, stride=2)
        x = F.relu(self.bn3(self.conv3(x)))
        x = F.max_pool2d(F.relu(self.bn4(self.conv4(x))), kernel_size=2, stride=2)
        x = x.view(x.size(0), -1)
        x = F.relu(self.bn5(self.dense1(x)))
        x = F.relu(self.bn6(self.dense2(x)))
        x = self.dense3(x)
        return x
    
class BasicBlock(nn.Module):
    def __init__(self, in_planes, out_planes, stride, dropRate=0.0):
        super(BasicBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.relu1 = nn.ReLU(inplace=True)
        self.conv1 = MaskedConv2d(in_planes, out_planes, kernel_size=(3, 3), stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_planes)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv2 = MaskedConv2d(out_planes, out_planes, kernel_size=(3, 3), stride=1,
                               padding=1, bias=False)
        self.droprate = dropRate
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = (not self.equalInOut) and MaskedConv2d(in_planes, out_planes, kernel_size=(1, 1), stride=stride,
                               padding=0, bias=False) or None
    def forward(self, x):
        if not self.equalInOut:
            x = self.relu1(self.bn1(x))
        else:
            out = self.relu1(self.bn1(x))
        out = self.relu2(self.bn2(self.conv1(out if self.equalInOut else x)))
        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)
        out = self.conv2(out)
        return torch.add(x if self.equalInOut else self.convShortcut(x), out)

class NetworkBlock(nn.Module): #it makes a Basicblock above for the given input
    def __init__(self, nb_layers, in_planes, out_planes, block, stride, dropRate=0.0):
        super(NetworkBlock, self).__init__()
        self.layer = self._make_layer(block, in_planes, out_planes, nb_layers, stride, dropRate)
    def _make_layer(self, block, in_planes, out_planes, nb_layers, stride, dropRate):
        layers = []
        for i in range(int(nb_layers)):
            layers.append(block(i == 0 and in_planes or out_planes, out_planes, i == 0 and stride or 1, dropRate))
        return nn.Sequential(*layers)
    def forward(self, x):
        return self.layer(x)

class MaskedWideResNet(nn.Module):
    def __init__(self, depth, num_classes, widen_factor=1, dropRate=0.0):
        super(MaskedWideResNet, self).__init__()
        nChannels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        assert((depth - 4) % 6 == 0)
        n = (depth - 4) / 6
        block = BasicBlock
        # 1st conv before any network block
        self.conv1 = MaskedConv2d(3, nChannels[0], kernel_size=(3, 3), stride=1,
                               padding=1, bias=False)
        # 1st block
        self.block1 = NetworkBlock(n, nChannels[0], nChannels[1], block, 1, dropRate)
        # 2nd block
        self.block2 = NetworkBlock(n, nChannels[1], nChannels[2], block, 2, dropRate)
        # 3rd block
        self.block3 = NetworkBlock(n, nChannels[2], nChannels[3], block, 2, dropRate)
        # global average pooling and classifier
        self.bn1 = nn.BatchNorm2d(nChannels[3])
        self.relu = nn.ReLU(inplace=True)
        self.fc = MaskedMLP(nChannels[3], num_classes)
        self.nChannels = nChannels[3]

        for m in self.modules():
            if isinstance(m, MaskedConv2d) or isinstance(m, MaskedMLP):
                m.reset_parameters()
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            
    def forward(self, x):
        out = self.conv1(x)
        out = self.block1(out)
        out = self.block2(out)
        out = self.block3(out)
        out = self.relu(self.bn1(out))
        out = F.avg_pool2d(out, 8)
        out = out.view(-1, self.nChannels)
        return self.fc(out)



class BasicBlock0(nn.Module):
    def __init__(self, in_planes, out_planes, stride, dropRate=0.0):
        super(BasicBlock0, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.relu1 = nn.ReLU(inplace=True)
        self.conv1 = nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_planes)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(out_planes, out_planes, kernel_size=3, stride=1,
                               padding=1, bias=False)
        self.droprate = dropRate
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = (not self.equalInOut) and nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride,
                               padding=0, bias=False) or None
    def forward(self, x):
        if not self.equalInOut:
            x = self.relu1(self.bn1(x))
        else:
            out = self.relu1(self.bn1(x))
        out = self.relu2(self.bn2(self.conv1(out if self.equalInOut else x)))
        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)
        out = self.conv2(out)
        return torch.add(x if self.equalInOut else self.convShortcut(x), out)

class NetworkBlock0(nn.Module):
    def __init__(self, nb_layers, in_planes, out_planes, block, stride, dropRate=0.0):
        super(NetworkBlock0, self).__init__()
        self.layer = self._make_layer(block, in_planes, out_planes, nb_layers, stride, dropRate)
    def _make_layer(self, block, in_planes, out_planes, nb_layers, stride, dropRate):
        layers = []
        for i in range(int(nb_layers)):
            layers.append(block(i == 0 and in_planes or out_planes, out_planes, i == 0 and stride or 1, dropRate))
        return nn.Sequential(*layers)
    def forward(self, x):
        return self.layer(x)

class WideResNet(nn.Module):
    def __init__(self, depth, num_classes, widen_factor=1, dropRate=0.0):
        super(WideResNet, self).__init__()
        nChannels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        assert((depth - 4) % 6 == 0)
        n = (depth - 4) / 6
        block = BasicBlock0
        # 1st conv before any network block
        self.conv1 = nn.Conv2d(3, nChannels[0], kernel_size=3, stride=1,
                               padding=1, bias=False)
        # 1st block
        self.block1 = NetworkBlock0(n, nChannels[0], nChannels[1], block, 1, dropRate)
        # 2nd block
        self.block2 = NetworkBlock0(n, nChannels[1], nChannels[2], block, 2, dropRate)
        # 3rd block
        self.block3 = NetworkBlock0(n, nChannels[2], nChannels[3], block, 2, dropRate)
        # global average pooling and classifier
        self.bn1 = nn.BatchNorm2d(nChannels[3])
        self.relu = nn.ReLU(inplace=True)
        self.fc = nn.Linear(nChannels[3], num_classes)
        self.nChannels = nChannels[3]

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.Linear):
                m.bias.data.zero_()
    def forward(self, x):
        out = self.conv1(x)
        out = self.block1(out)
        out = self.block2(out)
        out = self.block3(out)
        out = self.relu(self.bn1(out))
        out = F.avg_pool2d(out, 8)
        out = out.view(-1, self.nChannels)
        return self.fc(out)
